/*
 * Copyright 2004-2018 H2 Group. Multiple-Licensed under the MPL 2.0,
 * and the EPL 1.0 (http://h2database.com/html/license.html).
 * Initial Developer: H2 Group
 */
package org.h2.expression;

import org.h2.engine.Session;
import org.h2.table.ColumnResolver;
import org.h2.table.TableFilter;
import org.h2.value.Value;
import org.h2.value.ValueNull;

/**
 * Unary operation. Only negation operation is currently supported.
 */
public class UnaryOperation extends Expression {

    private Expression arg;
    private int dataType;

    public UnaryOperation(Expression arg) {
        this.arg = arg;
    }

    @Override
    public String getSQL() {
        // don't remove the space, otherwise it might end up some thing like
        // --1 which is a line remark
        return "(- " + arg.getSQL() + ')';
    }

    @Override
    public Value getValue(Session session) {
        Value a = arg.getValue(session).convertTo(dataType, session.getDatabase().getMode());
        return a == ValueNull.INSTANCE ? a : a.negate();
    }

    @Override
    public void mapColumns(ColumnResolver resolver, int level) {
        arg.mapColumns(resolver, level);
    }

    @Override
    public Expression optimize(Session session) {
        arg = arg.optimize(session);
        dataType = arg.getType();
        if (dataType == Value.UNKNOWN) {
            dataType = Value.DECIMAL;
        } else if (dataType == Value.ENUM) {
            dataType = Value.INT;
        }
        if (arg.isConstant()) {
            return ValueExpression.get(getValue(session));
        }
        return this;
    }

    @Override
    public void setEvaluatable(TableFilter tableFilter, boolean b) {
        arg.setEvaluatable(tableFilter, b);
    }

    @Override
    public int getType() {
        return dataType;
    }

    @Override
    public long getPrecision() {
        return arg.getPrecision();
    }

    @Override
    public int getDisplaySize() {
        return arg.getDisplaySize();
    }

    @Override
    public int getScale() {
        return arg.getScale();
    }

    @Override
    public void updateAggregate(Session session, boolean window) {
        arg.updateAggregate(session, window);
    }

    @Override
    public boolean isEverything(ExpressionVisitor visitor) {
        return arg.isEverything(visitor);
    }

    @Override
    public int getCost() {
        return arg.getCost() + 1;
    }

    @Override
    public int getSubexpressionCount() {
        return 1;
    }

    @Override
    public Expression getSubexpression(int index) {
        if (index == 0) {
            return arg;
        }
        throw new IndexOutOfBoundsException();
    }

}
